/*
 * MIT License
 *
 * Copyright (c) 2002-2023 Mikko Tommila
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.apfloat.internal;

import org.apfloat.*;
import org.apfloat.spi.*;

/**
 * @version 1.9.0
 * @author Mikko Tommila
 */

public abstract class DoubleNTTStrategyTestCase
    extends DoubleTestCase
    implements DoubleModConstants
{
    protected DoubleNTTStrategyTestCase(String methodName)
    {
        super(methodName);
    }

    protected static DataStorage createDataStorage(int size)
    {
        ApfloatContext ctx = ApfloatContext.getContext();
        DataStorageBuilder dataStorageBuilder = ctx.getBuilderFactory().getDataStorageBuilder();
        DataStorage dataStorage = dataStorageBuilder.createDataStorage(size * Double.BYTES);
        dataStorage.setSize(size);

        try (ArrayAccess arrayAccess = dataStorage.getArray(DataStorage.WRITE, 0, size))
        {
            for (int i = 0; i < size; i++)
            {
                arrayAccess.getDoubleData()[arrayAccess.getOffset() + i] = (double) (i + 1);
            }
        }

        return dataStorage;
    }

    protected static double[] getPlainArray(DataStorage dataStorage)
    {
        int size = (int) dataStorage.getSize();
        double[] data = new double[size];
        try (ArrayAccess arrayAccess = dataStorage.getArray(DataStorage.READ, 0, size))
        {
            System.arraycopy(arrayAccess.getDoubleData(), arrayAccess.getOffset(), data, 0, size);
        }

        return data;
    }

    protected static double[] ntt(double[] data, int modulus)
    {
        DoubleModMath math = new DoubleModMath();
        math.setModulus(MODULUS[modulus]);

        double[] transform = new double[data.length];
        double w = math.getForwardNthRoot(PRIMITIVE_ROOT[modulus], data.length),
                wi = (double) 1;

        for (int i = 0; i < data.length; i++)
        {
            double wj = (double) 1;

            for (int j = 0; j < data.length; j++)
            {
                transform[i] = math.modAdd(transform[i], math.modMultiply(wj, data[j]));
                wj = math.modMultiply(wj, wi);
            }

            wi = math.modMultiply(wi, w);
        }

        return transform;
    }

    protected static double[] inverseNtt(double[] data, int modulus)
    {
        DoubleModMath math = new DoubleModMath();
        math.setModulus(MODULUS[modulus]);

        double[] transform = new double[data.length];
        double w = math.getInverseNthRoot(PRIMITIVE_ROOT[modulus], data.length),
                wi = (double) 1;

        for (int i = 0; i < data.length; i++)
        {
            double wj = (double) 1;

            for (int j = 0; j < data.length; j++)
            {
                transform[i] = math.modAdd(transform[i], math.modMultiply(wj, data[j]));
                wj = math.modMultiply(wj, wi);
            }

            transform[i] = math.modDivide(transform[i], (double) data.length);

            wi = math.modMultiply(wi, w);
        }

        return transform;
    }

    protected static void runRoundTrip(NTTStrategy nttStrategy, int size)
    {
        DataStorage dataStorage = createDataStorage(size + 5).subsequence(5, size);

        for (int modulus = 0; modulus < 3; modulus++)
        {
            nttStrategy.transform(dataStorage, modulus);

            DataStorage.Iterator iterator = dataStorage.iterator(DataStorage.READ, 0, 1);

            assertTrue("transformed [0]", 6 != (long) iterator.getDouble());
            iterator.close();

            nttStrategy.inverseTransform(dataStorage, modulus, size);

            iterator = dataStorage.iterator(DataStorage.READ, 0, size);

            for (int i = 0; i < size; i++)
            {
                assertEquals("round-tripped [" + i + "]", i + 6, (long) iterator.getDouble());
                iterator.next();
            }
        }
    }
}
