/*
 * MIT License
 *
 * Copyright (c) 2002-2023 Mikko Tommila
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.apfloat.internal;

import static org.junit.Assert.assertArrayEquals;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.lang.reflect.Method;

import junit.framework.TestSuite;

/**
 * @version 1.9.0
 * @author Mikko Tommila
 */

public class IntCRTMathTest
    extends IntTestCase
    implements IntModConstants
{
    public IntCRTMathTest(String methodName)
    {
        super(methodName);
    }

    public static void main(String[] args)
    {
        junit.textui.TestRunner.run(suite());
    }

    public static TestSuite suite()
    {
        TestSuite suite = new TestSuite();

        suite.addTest(new IntCRTMathTest("testMultiply"));
        suite.addTest(new IntCRTMathTest("testCompare"));
        suite.addTest(new IntCRTMathTest("testAdd"));
        suite.addTest(new IntCRTMathTest("testSubtract"));
        suite.addTest(new IntCRTMathTest("testDivide"));
        suite.addTest(new IntCRTMathTest("testSerialization"));

        return suite;
    }

    public static void testMultiply()
    {
        int b1 = MAX_POWER_OF_TWO_BASE - (int) 1;
        int[] src = { b1, b1 };
        int[] dst = new int[3];

        new IntCRTMath(2).multiply(src, b1, dst);

        assertEquals("max[0]", (long) b1 - 1, (long) dst[0]);
        assertEquals("max[1]", (long) b1, (long) dst[1]);
        assertEquals("max[2]", (long) 1, (long) dst[2]);

        src = new int[] { (int) 2, (int) 4 };

        new IntCRTMath(2).multiply(src, (int) 3, dst);

        assertEquals("normal[0]", 0, (long) dst[0]);
        assertEquals("normal[1]", 6, (long) dst[1]);
        assertEquals("normal[2]", 12, (long) dst[2]);
    }

    public static void testCompare()
    {
        int b1 = MAX_POWER_OF_TWO_BASE - (int) 1;
        int result = new IntCRTMath(2).compare(new int[] { (int) 1, (int) 1, (int) 1 },
                                                new int[] { (int) 2, (int) 1, (int) 1 });
        assertTrue("1st", result < 0);

        result = new IntCRTMath(2).compare(new int[] { (int) 1, (int) 2, (int) 1 },
                                                new int[] { (int) 1, (int) 1, (int) 1 });
        assertTrue("2nd", result > 0);

        result = new IntCRTMath(2).compare(new int[] { (int) 1, (int) 1, (int) 0 },
                                                new int[] { (int) 1, (int) 1, (int) b1 });
        assertTrue("3rd", result < 0);

        result = new IntCRTMath(2).compare(new int[] { (int) 1, (int) 1, (int) 1 },
                                                new int[] { (int) 1, (int) 1, (int) 1 });
        assertTrue("equal", result == 0);
    }

    public static void testAdd()
    {
        int b1 = MAX_POWER_OF_TWO_BASE - (int) 1;
        int[] src = { b1, b1, b1 };
        int[] srcDst = { b1, b1, b1 };

        int carry = new IntCRTMath(2).add(src, srcDst);

        assertEquals("max carry", 1, (long) carry);
        assertEquals("max[0]", (long) b1, (long) srcDst[0]);
        assertEquals("max[1]", (long) b1, (long) srcDst[1]);
        assertEquals("max[2]", (long) b1 - 1, (long) srcDst[2]);

        src = new int[] { (int) 2, (int) 4, (int) 6 };
        srcDst = new int[] { (int) 3, (int) 5, (int) 7 };

        carry = new IntCRTMath(2).add(src, srcDst);

        assertEquals("normal carry", 0, (long) carry);
        assertEquals("normal[0]", 5, (long) srcDst[0]);
        assertEquals("normal[1]", 9, (long) srcDst[1]);
        assertEquals("normal[2]", 13, (long) srcDst[2]);
    }

    public static void testSubtract()
    {
        int b1 = MAX_POWER_OF_TWO_BASE - (int) 1;
        int[] src = { b1, b1, b1 };
        int[] srcDst = { b1, b1, b1 };

        new IntCRTMath(2).subtract(src, srcDst);

        assertEquals("max[0]", 0, (long) srcDst[0]);
        assertEquals("max[1]", 0, (long) srcDst[1]);
        assertEquals("max[2]", 0, (long) srcDst[2]);

        src = new int[] { 0, 0, (int) 1 };
        srcDst = new int[] { (int) 1, 0, 0 };

        new IntCRTMath(2).subtract(src, srcDst);

        assertEquals("normal[0]", 0, (long) srcDst[0]);
        assertEquals("normal[1]", (long) b1, (long) srcDst[1]);
        assertEquals("normal[2]", (long) b1, (long) srcDst[2]);
    }

    public static void testDivide()
    {
        int[] srcDst = new int[] { (int) 1, 0, 1 };

        int remainder = new IntCRTMath(2).divide(srcDst);

        assertEquals("normal remainder", 1, (long) remainder);
        assertEquals("normal[0]", 0, (long) srcDst[0]);
        assertEquals("normal[1]", 2, (long) srcDst[1]);
        assertEquals("normal[2]", 0, (long) srcDst[2]);
    }

    public void testSerialization()
        throws Exception
    {
        if (Integer.TYPE.getName().equals("long"))
        {
            String className = IntCRTMath.class.getName();
            Java9ClassLoader classLoader = new Java9ClassLoader(getClass().getClassLoader());
            classLoader.loadJava9Class(IntBaseMath.class.getName()); // Load base class Java 9 specific version also
            Class<?> crtMathClass = classLoader.loadJava9Class(className);
            Object crtMath = crtMathClass.getConstructor(Integer.TYPE).newInstance(2);
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            try (ObjectOutputStream out = new ObjectOutputStream(buffer))
            {
                out.writeObject(crtMath);
            }
            byte[] java9Data = buffer.toByteArray();
            buffer = new ByteArrayOutputStream();
            try (ObjectOutputStream out = new ObjectOutputStream(buffer))
            {
                out.writeObject(new IntCRTMath(2));
            }
            byte[] java8Data = buffer.toByteArray();
            assertArrayEquals("Serialized data", java8Data, java9Data);

            ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(java8Data))
            {
                @Override
                public Class<?> resolveClass(ObjectStreamClass desc)
                    throws IOException, ClassNotFoundException
                {
                    String name = desc.getName();
                    return classLoader.loadClass(name);
                }
            };
            crtMath = in.readObject();
            Method method = crtMath.getClass().getMethod("divide", int[].class);
            int[] srcDst =  { (int) 1, (int) 0, (int) 1 };
            int remainder = (Integer) method.invoke(crtMath, srcDst);
            assertEquals("Deserialized java 9 classloader", classLoader, crtMath.getClass().getClassLoader());
            assertEquals("Deserialized java 9 remainder", (int) 1, remainder);
            assertEquals("Deserialized java 9 result [0]", (int) 0, srcDst[0]);
            assertEquals("Deserialized java 9 result [1]", (int) 2, srcDst[1]);
            assertEquals("Deserialized java 9 result [2]", (int) 0, srcDst[2]);
        }
    }
}
